/*:
 * @target MZ
 * @plugindesc v1.3 (patched v5: unequip weapons immediately + refresh windows)
 * @author You
 *
 * @help EquipClassSwitchMZ.js
 * 装備品（武器/防具）のメモ欄に <ClassChange:2> のように書くと、
 * その装備を着けている間、アクターの職業を ID=2 に切り替えます。
 * ClassChangeタグを持つ装備が1つも無い場合は「デフォルト職業ID」に戻します。
 *
 * ● 複数装備したら？
 *   スロット番号の小さい装備（上のスロット）を優先します。
 *
 * ● レベルの扱い（切替時のレベル挙動）
 *   - perClassLevel（既定）：職業ごとにレベルを記憶し、再装備で復元（無ければFallbackLevel）。
 *   - preserveActorLevel：現在のレベルをそのまま引き継ぎ。
 *   - fixed1：常にレベル1に設定。
 *
 * ● タグの付け方（例）
 *   剣士の指輪のメモ欄に： <ClassChange:2>
 *
 * ● 互換
 *   - Game_Actor.changeEquip 等の装備操作をフックし、都度職業再判定。
 *   - _data構造は触らず、描画系は変更しません（他プラグインと共存しやすい）。
 *
 * @param DefaultClassId
 * @text デフォルト職業ID（該当装備なし時）
 * @type class
 * @default 1
 *
 * @param SwitchLevelBehavior
 * @text レベル挙動
 * @type select
 * @option perClassLevel（職業ごとに記憶&復元）
 * @value perClassLevel
 * @option preserveActorLevel（現在Lvを維持）
 * @value preserveActorLevel
 * @option fixed1（常にLv1）
 * @value fixed1
 * @default perClassLevel
 *
 * @param FallbackLevel
 * @text 初回切替の初期レベル
 * @type number
 * @min 1
 * @desc perClassLevelで、過去に記録が無い職業へ初めて切り替えるときのレベル
 * @default 1
 *
 * @param ShowSwitchToast
 * @text 職業切替トースト表示
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default false
 *
 * @param ToastFormat
 * @text トースト書式
 * @type string
 * @desc 例：\C[6]職業変更：\C[0]%1（Lv%2） ※%1=職業名, %2=レベル
 * @default \\C[6]職業変更：\\C[0]%1（Lv%2）
 */
(() => {
  'use strict';

  const PLUGIN_NAME = 'EquipClassSwitchMZ';
  const P = PluginManager.parameters(PLUGIN_NAME);

  const DEFAULT_CLASS_ID = Number(P['DefaultClassId'] || 1);
  const SWITCH_BEHAVIOR = String(P['SwitchLevelBehavior'] || 'perClassLevel'); // perClassLevel | preserveActorLevel | fixed1
  const FALLBACK_LEVEL = Math.max(1, Number(P['FallbackLevel'] || 1));
  const SHOW_TOAST = P['ShowSwitchToast'] === 'true';
  const TOAST_FORMAT = String(P['ToastFormat'] || '\\C[6]職業変更：\\C[0]%1（Lv%2）');

  // アクターに職業別レベル記録フィールドを追加
  const LEVEL_MAP_KEY = '_ecsClassLevels';

  function ensureLevelMap(actor) {
    if (!actor[LEVEL_MAP_KEY]) actor[LEVEL_MAP_KEY] = {};
    return actor[LEVEL_MAP_KEY];
  }

  // アイテムのメタから ClassChange を読む
  function itemClassTag(item) {
    if (!item || !item.meta) return null;
    const v = item.meta.ClassChange || item.meta['ClassChange'];
    if (v == null) return null;
    const n = Number(v);
    return Number.isFinite(n) && n > 0 ? n : null;
  }

  // 装備から優先クラスIDを決定（スロット番号昇順優先）
  function classIdFromEquips(actor) {
    const equips = actor.equips(); // [slot0, slot1, ...]
    for (let i = 0; i < equips.length; i++) {
      const it = equips[i];
      const clsId = itemClassTag(it);
      if (clsId) return { classId: clsId, slotId: i, item: it };
    }
    return { classId: null, slotId: -1, item: null };
  }

  function classNameById(classId) {
    const c = $dataClasses[classId];
    return c ? c.name : String(classId);
  }



  // 武器（装備タイプ=1）を全解除してパーティへ戻す
  // ※クラス切替に使った装備が「武器」だった場合は、その武器だけ残す（即座に外れて職業が戻るのを防ぐ）
  function unequipAllWeapons(actor, exceptSlotId = -1) {
    if (!actor) return;
    const slots = actor.equipSlots(); // 装備タイプ配列（NRP_EquipSlotなどで増えていてもOK）
    for (let i = 0; i < slots.length; i++) {
      if (i === exceptSlotId) continue;
      if (slots[i] === 1) { // 1 = weapon etypeId
        actor.changeEquip(i, null);
      }
    }
  }

  function switchClassIfNeeded(actor) {
    if (!actor) return;

    // 1) 装備からクラスID決定（クラス切替に使ったスロットも取る）
    const tagged = classIdFromEquips(actor);

    // 2) 対象クラスID
    const targetClassId = (tagged.classId != null) ? tagged.classId : DEFAULT_CLASS_ID;

    // すでに同じなら何もしない
    const beforeClassId = actor.currentClass() ? actor.currentClass().id : null;
    if (beforeClassId === targetClassId) return;

    // 3) レベル挙動の決定
    const levelMap = ensureLevelMap(actor);
    let targetLevel = null;

    if (SWITCH_BEHAVIOR === 'preserveActorLevel') {
      targetLevel = actor.level; // 現在Lvそのまま
    } else if (SWITCH_BEHAVIOR === 'fixed1') {
      targetLevel = 1;
    } else { // perClassLevel
      // 現在の職業のLvを記録
      const curClass = actor.currentClass();
      if (curClass && curClass.id) {
        levelMap[curClass.id] = actor.level;
      }
      // 目標職業の記録があれば復元、無ければフォールバック
      targetLevel = levelMap[targetClassId] || FALLBACK_LEVEL;
    }

    // 4) 職業切替
    const keepExp = false; // 経験値は使わず、明示的にレベルを合わせる
    actor.changeClass(targetClassId, keepExp);
    if (targetLevel != null) {
      actor.changeLevel(Math.max(1, targetLevel), false);
    }
    actor.refresh();

    // 5) 職業変更装備を付け替えた直後は、武器スロットの再構成と表示がズレやすいので
    //    武器（etypeId=1）を全解除して確実に整合させる。
    //    ※クラス切替に使った装備が武器の場合は、そのスロットだけ残す。
    unequipAllWeapons(actor, (tagged.item && DataManager.isWeapon(tagged.item)) ? tagged.slotId : -1);

    // 6) 不適合装備を即解除して反映
    actor.releaseUnequippableItems(true);
    actor.refresh();
    // 5) トースト
    if (SHOW_TOAST && SceneManager._scene && SceneManager._scene instanceof Scene_Map) {
      const name = classNameById(targetClassId);
      const text = TOAST_FORMAT.replace('%1', name).replace('%2', String(actor.level));
      $gameMessage.add(text);
    }
  }

  // ---- 装備操作のフック ----
  const _Game_Actor_changeEquip = Game_Actor.prototype.changeEquip;
  Game_Actor.prototype.changeEquip = function(slotId, item) {
    _Game_Actor_changeEquip.call(this, slotId, item);
    // 装備画面では比較プレビューが走るので、確定時（Scene_Equip.onItemOk）に判定する。
    if (!(SceneManager._scene && SceneManager._scene instanceof Scene_Equip)) {
      switchClassIfNeeded(this);
    }
  };
  const _Game_Actor_changeEquipById = Game_Actor.prototype.changeEquipById;
  Game_Actor.prototype.changeEquipById = function(etypeId, itemId) {
    _Game_Actor_changeEquipById.call(this, etypeId, itemId);
    if (!(SceneManager._scene && SceneManager._scene instanceof Scene_Equip)) {
      switchClassIfNeeded(this);
    }
  };

  // セーブデータからの復元後などでの保険（マップ突入時に一度判定）
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    $gameParty.members().forEach(a => switchClassIfNeeded(a));
  };

  // アクター変更時の保険（メニューで装備画面を開いた直後など）
  const _Scene_Equip_create = Scene_Equip.prototype.create;
  Scene_Equip.prototype.create = function() {
    _Scene_Equip_create.call(this);
    const actor = this.actor();
    if (actor) switchClassIfNeeded(actor);
  };

  // ---- 装備画面での「確定」時にのみ職業判定（プレビューでは判定しない） ----
  const _Scene_Equip_onItemOk = Scene_Equip.prototype.onItemOk;
  Scene_Equip.prototype.onItemOk = function() {
    _Scene_Equip_onItemOk.call(this);
    const actor = this.actor();
    if (actor) {
      switchClassIfNeeded(actor);
      // 画面内に即反映（武器解除など）
      actor.releaseUnequippableItems(true);
      actor.refresh();
    }
    // ウィンドウを即更新
    if (this._slotWindow) this._slotWindow.refresh();
    if (this._itemWindow) this._itemWindow.refresh();
    if (this._statusWindow) this._statusWindow.refresh();
  };

  // 装備画面を閉じるタイミングでも念のため最終判定
  const _Scene_Equip_terminate = Scene_Equip.prototype.terminate;
  Scene_Equip.prototype.terminate = function() {
    _Scene_Equip_terminate.call(this);
    const actor = this.actor && this.actor();
    if (actor) switchClassIfNeeded(actor);
  };

})();
